def quit(status):
    from sys import exit

    print 'Press any key to quit.'
    try: input = raw_input()
    except NameError: pass
    exit(status)

class CacheSim():

    def __str__(self):
        return 'CacheSim'

    def __init__(self, cache):

        import logging
        self.log = logging.getLogger(str(self))
        
        self.cache = cache

    def readTrace(self, traceFile):
        '''Read a CPU cache trace file'''

        self.log.debug('  Beginning trace file read')

        labels, addresses = [], []
        lines = None

        try: 
            with open(traceFile, 'r') as f: 
                lines = list(f)   # CDS: Read the cache trace file

        except IOError as e:
            self.log.critical(e)
            self.log.critical('    Could not open trace file \'%s\'' % traceFile)
            quit(1)

        for i, line in enumerate(lines): # CDS: For each line in the trace
            words = lines[i].split()    # CDS: Split the line into words
            try:
                labels.append(int(words[0]))     # CDS: Get the label (first word)
                addresses.append(int(words[1], 16))  # CDS: Get the hexadecimal (base-16) address (second word)

            except IndexError: continue     # CDS: A label or address is missing

        self.log.debug('  Finished trace file read')

        return labels, addresses

    def doSim(self, trace):
        '''Given a CPU cache trace, perform the cache simulation'''

        import itertools

        self.log.debug('Beginning simulation')

        labels, addresses = self.readTrace(trace)

        readAccesses, writeAccesses, ifAccesses = 0, 0, 0
        readHits, writeHits, ifHits = 0, 0, 0
        accessTimes = []
        
        for label, address in itertools.izip(labels, addresses):

            read = label == self.cache.cmds.RD # True if the label designates a read
            write = label == self.cache.cmds.WR # True if the label designates a write
            ifetch = label == self.cache.cmds.IF # True if the label designates an instruction fetch

            accessTime = 0
            if read: 
                readAccesses += 1
                hit, accessTime = self.cache.read(address)
                if hit: readHits += 1
            elif write: 
                writeAccesses += 1
                hit, accessTime = self.cache.write(address)
                if hit: writeHits += 1
            elif ifetch: 
                ifAccesses += 1
                hit, accessTime = self.cache.instructionFetch(address)
                if hit: ifHits += 1
            
            accessTimes.append(accessTime)

            # Output: "[label] [address] [cycles]", e.g. "0 fff0 529"
            #self.log.debug('%d %x %d' % (label, address, self.cache.clock.cycle))

        readMisses = readAccesses - readHits
        writeMisses = writeAccesses - writeHits
        ifMisses = ifAccesses - ifHits

        totalHits = sum([readHits, writeHits, ifHits])
        totalMisses = sum([readMisses, writeMisses, ifMisses])
        totalAccesses = totalHits + totalMisses

        assert totalAccesses == sum([readAccesses, writeAccesses, ifAccesses])

        avgAccessTime = sum(accessTimes)/float(len(accessTimes))
        self.log.debug('Finished at clock cycle %d' % self.cache.clock.currentCycle)
        self.log.debug('Avg access time %f' % avgAccessTime)
        self.log.debug('Read Hits: %d Write Hits: %d Instruction Fetch Hits: %d Total Hits: %d'
                       % (readHits, writeHits, ifHits, totalHits))
        self.log.debug('Read Misses: %d Write Misses: %d Instruction Fetch Misses: %d Total Misses: %d'
                       % (readMisses, writeMisses, ifMisses, totalMisses))
        self.log.debug('Total accesses: %d' % totalAccesses)

def parseArgs():
    '''Parse command line arguments'''

    import logging, optparse
    from sys import stdout

    # Define logging detail levels

    LOGGING_LEVELS = {'critical': logging.CRITICAL,
                  'error': logging.ERROR,
                  'warning': logging.WARNING,
                  'info': logging.INFO,
                  'debug': logging.DEBUG }

    # Create command line options and their defaults.
    parser = optparse.OptionParser()
    parser.add_option('-l', '--logging-level', help='Logging level', default = 'debug')
    parser.add_option('-f', '--logging-file', help='Logging file name', default = 'cacheSimLog.txt')
    parser.add_option('-t', '--trace-file', help='Trace file name (typically something.din)', default = '../traces/sample.din' )
    parser.add_option('-b', '--blockSize', help='Cache block size as power of 2', default = '2')
    parser.add_option('-s', '--cacheSize', help='Cache size in kB', default = '1')
    parser.add_option('-p', '--writePolicy', help='Cache write policy = { writeback | writethrough }', default = 'writeback')
    parser.add_option('-w', '--writeAllocate', help='Write allocate = { 1 = true | 0 = false }', default = '1')
    parser.add_option('-m', '--memPenalty', help='Memory access penalty in clock cyles (write and read uniform)', default = '4')
    parser.add_option('-a', '--associativity', help='Cache associativity = { 1, 2, 4, 8 }', default = '1')
    (options, args) = parser.parse_args()

    # Print to log file
    logFormat='%(asctime)s %(levelname)s: %(message)s'
    logging_level = LOGGING_LEVELS.get(options.logging_level, logging.NOTSET)
    logging.basicConfig(level=logging_level, filename=options.logging_file,
                      format=logFormat, datefmt='%Y-%m-%d %H:%M:%S')  

    # Also print to stdout
    console = logging.StreamHandler(stdout)
    console.setLevel(LOGGING_LEVELS.get(options.logging_level, logging.NOTSET))
    console.setFormatter(logging.Formatter(logFormat))
    logging.getLogger().addHandler(console)
    
    return options          

def main():
    import Cache, math
    
    options = parseArgs()

    cacheSize = int(options.cacheSize)
    blockSize  = int(math.pow(2,int(options.blockSize))) # 2^blockSize
    associativity = int(options.associativity)
    policy = options.writePolicy
    allocate = int(options.writeAllocate)
    memPenalty = int(options.memPenalty)

    cache = Cache.DataCache(cacheSize, blockSize, associativity, policy, allocate, 1, memPenalty)
    sim = CacheSim(cache)

    trace = options.trace_file
    sim.doSim(trace)

if __name__ == "__main__":
    main()

